/*
 * ProjectileAttackInfo.h
 *
 * Created 9/6/2009 By Johnny Huynh
 *
 * Version 00.00.01 9/6/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 /**
  * Only CombatCharacter type Objects can project an attack. Make sure to supply a 
  * CombatCharacter type when creating an ActionTask or EffectTask using
  * a ProjectileAttackInfo.
  */
 
 #ifndef PROJECTILE_ATTACK_INFO_H
 #define PROJECTILE_ATTACK_INFO_H
 
 template <typename T> class ProjectileAttackInfo;
 
 #include "global.h"
 
 #include "AttackInfo.h"
 #include "MoveInfo.h"
 #include "DamageObject.h"
 #include "pointerTo.h"
 
 /**
  * Class specification for ProjectileAttackInfo
  */
 template <typename T>
 class ProjectileAttackInfo : public AttackInfo<T>
 {
 // Private Static Functions
 private:
    static inline AsyncTask::DoneStatus process_projectile_attack( GenericAsyncTask* task_Ptr, void* data_Ptr );
    static inline void upon_death_handler( GenericAsyncTask* task_Ptr, bool clean_exit, void* data_Ptr );
    
 // Data Members
 private:
    NodePath _dmg_obj_parent; // no need for this; should probably remove
    MovementInfoCollection<T> _proj_movement_info_collection; // contains movement information for the DamageObject (i.e. projectile)
    
 // Local Functions
 public:
    ProjectileAttackInfo( const NodePath& dmg_obj_parent = global::_world_Ptr->get_scene_graph_root_node_path(), 
                          DamageObject<T>* dmg_obj_Ptr = NULL, 
                          const double duration_before_attaching_dmg_obj = 0.0, 
                          const double duration_before_detaching_dmg_obj = 0.0, // set to negative number if dmg_obj should not be detached
                          const MovementInfoCollection<T>& movement_info_collection = MovementInfoCollection<T>() );
    ProjectileAttackInfo( const ProjectileAttackInfo<T>& projectile_attack_info );
    virtual ~ProjectileAttackInfo();
    inline ProjectileAttackInfo<T>& operator=( const ProjectileAttackInfo<T>& projectile_attack_info );
    virtual inline void split_tasks( ActionTask<T>* action_task_Ptr, AsyncTaskManager* task_mgr_Ptr );
 
 // Private Functions
 private:
    
 // Public Static Functions
 public:
    
 };
 
 /** PRIVATE STATIC FUNCTIONS **/
 
 /**
  * process_projectile_attack() handles attaching the DamageObject to its parent.
  * CAVEAT: The task pointed to by the specified task_Ptr must be an ActionTask.
  * The ActionInfo of the ActionTask must also be a ProjectileAttackInfo.
  *
  * @param (GenericAsyncTask*) task_Ptr
  * @param (void*) data_Ptr
  * @return AsyncTask::DoneStatus
  */
 template <typename T>
 inline AsyncTask::DoneStatus ProjectileAttackInfo<T>::process_projectile_attack( GenericAsyncTask* task_Ptr, void* data_Ptr )
 {
    nassertr( task_Ptr != NULL, AsyncTask::DS_done );
    
    PT(ActionTask<T>) action_task_Ptr( reinterpret_cast<ActionTask<T>*>( task_Ptr ) );
    PT(ProjectileAttackInfo<T>) proj_attack_info_Ptr( dynamic_cast<ProjectileAttackInfo<T>*>( action_task_Ptr->get_action_info() ) );
    nassertr( proj_attack_info_Ptr != NULL, AsyncTask::DS_done );
    
    double elapsed_time( global::_clock_Ptr->get_real_time() - proj_attack_info_Ptr->get_time_action_was_invoked() );
    
    // if the elapsed time has reached for attaching the DamageObject
    if ( elapsed_time > proj_attack_info_Ptr->get_duration_before_attaching_damage_object() )
    {
        // if the parent is the MapArea, then add the DamageObject to the MapArea
        if ( proj_attack_info_Ptr->_dmg_obj_parent.get_key() == global::_world_Ptr->get_scene_graph_root_node_path().get_key() )
            global::_world_Ptr->get_map_area()->add_object( proj_attack_info_Ptr->get_damage_object() );
        else // simply attach the DamageObject to the parent
            proj_attack_info_Ptr->get_damage_object()->NodePath::reparent_to( proj_attack_info_Ptr->_dmg_obj_parent );
    }
    
    // if the elapsed time has reached for detaching the DamageObject
    if ( proj_attack_info_Ptr->get_duration_before_detaching_damage_object() > 0.0 
            && elapsed_time > proj_attack_info_Ptr->get_duration_before_detaching_damage_object() )
    {
        //proj_attack_info_Ptr->get_damage_object()->detach_node();
        return AsyncTask::DS_done;
    }
    
    return AsyncTask::DS_cont;
 }
 
 /**
  * upon_death_handler() detaches the DamageObject that may have originally been attached; the
  * tasks associated with the DamageObject are also removed.
  * CAVEAT: The task pointed to by the specified task_Ptr must be an ActionTask. The ActionInfo
  * of the ActionTask must also be an ProjectileAttackInfo.
  *
  * @param (GenericAsyncTask*) task_Ptr
  * @param (bool) clean_exit
  * @param (void*) data_Ptr
  */
 template <typename T>
 inline void ProjectileAttackInfo<T>::upon_death_handler( GenericAsyncTask* task_Ptr, bool clean_exit, void* data_Ptr )
 {
    nassertv( task_Ptr != NULL );
    
    PT(ActionTask<T>) action_task_Ptr( reinterpret_cast<ActionTask<T>*>( task_Ptr ) );
    PT(ProjectileAttackInfo<T>) proj_attack_info_Ptr( dynamic_cast<ProjectileAttackInfo<T>*>( action_task_Ptr->get_action_info() ) );
    nassertv( proj_attack_info_Ptr != NULL );
    
    // if the parent is the MapArea, then have the DamageObject destruct
    if ( proj_attack_info_Ptr->_dmg_obj_parent.get_key() == global::_world_Ptr->get_scene_graph_root_node_path().get_key() )
        global::_world_Ptr->get_map_area()->remove_object( proj_attack_info_Ptr->get_damage_object() );
    else // simply detach the DamageObject for it to destruct
        proj_attack_info_Ptr->get_damage_object()->detach_node();
 }
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 ProjectileAttackInfo<T>::ProjectileAttackInfo( const NodePath& dmg_obj_parent,
                                                DamageObject<T>* dmg_obj_Ptr, // should pass a copy of the damage object
                                                const double duration_before_attaching_dmg_obj, 
                                                const double duration_before_detaching_dmg_obj,
                                                const MovementInfoCollection<T>& proj_movement_info_collection )
               : AttackInfo<T>( dmg_obj_Ptr, duration_before_attaching_dmg_obj, duration_before_detaching_dmg_obj ),
                 _dmg_obj_parent( dmg_obj_parent ),
                 _proj_movement_info_collection( proj_movement_info_collection )
 {
    nassertv( !_dmg_obj_parent.is_empty() );
    nassertv( dmg_obj_Ptr != NULL );
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 ProjectileAttackInfo<T>::ProjectileAttackInfo( const ProjectileAttackInfo<T>& proj_attack_info )
                    : AttackInfo<T>( proj_attack_info ),
                      _proj_movement_info_collection( proj_attack_info._proj_movement_info_collection )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 ProjectileAttackInfo<T>::~ProjectileAttackInfo()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified ProjectileAttackInfo to this ProjectileAttackInfo.
  *
  * @param (const ProjectileAttackInfo<T>&) proj_attack_info
  * @return ProjectileAttackInfo<T>&
  */
 template <typename T>
 inline ProjectileAttackInfo<T>& ProjectileAttackInfo<T>::operator=( const ProjectileAttackInfo<T>& proj_attack_info )
 {
    AttackInfo<T>::operator=( proj_attack_info );
    _proj_movement_info_collection = proj_attack_info._proj_movement_info_collection;
    
    return *this;
 }
 
 /**
  * split_tasks() either assigns the specified task to the specified 
  * task manager or splits up the task into multiple tasks before 
  * assigning the tasks to the specified task manager.
  *
  * @param (ActionTask<T>*) action_task_Ptr
  * @param (AsyncTaskManager*) task_mgr_Ptr
  */
 template <typename T>
 inline void ProjectileAttackInfo<T>::split_tasks( ActionTask<T>* action_task_Ptr, AsyncTaskManager* task_mgr_Ptr )
 {
    nassertv( action_task_Ptr != NULL );
    nassertv( task_mgr_Ptr != NULL );
    
    // we use != 0.0 instead of > 0.0 because if the duration_before_detaching_damage_object is negative,
    // the DamageObject will not be detached after any specific duration (i.e. time will not determine when
    // the DamageObject will be detached from its parent)
    if ( AttackInfo<T>::get_duration_before_detaching_damage_object() != 0.0 )
    {
        // Create the ActionTasks to be added to the AsyncTaskManager
        PT(MoveInfo<T>) move_info_Ptr( new MoveInfo<T>( _proj_movement_info_collection, AttackInfo<T>::get_duration_before_attaching_damage_object() + AttackInfo<T>::get_time_action_was_invoked() ) );
        //PT(ActionTask<T>) move_action_task_Ptr( new ActionTask<T>( AttackInfo<T>::get_damage_object(), &MoveInfo<T>::process_move, move_info_Ptr ) );
        PT(ActionTask<T>) move_action_task_Ptr( new ActionTask<T>( AttackInfo<T>::get_damage_object(), move_info_Ptr ) );
        // would be more efficient to create and add this move_action_task_Ptr to the task manager 
        // in the process_projectile_attack function
        
        PT(ActionTask<T>) proj_action_task_Ptr( new ActionTask<T>( AttackInfo<T>::get_damage_object(), &ProjectileAttackInfo<T>::process_projectile_attack, action_task_Ptr->get_action_info() ) );
        proj_action_task_Ptr->set_upon_death( &ProjectileAttackInfo<T>::upon_death_handler );
        
        move_action_task_Ptr->add_to_task_manager(); // use this because &MoveInfo<T>::process_move is private and needs to be specified as the GenericAsyncTask function
        //move_action_task_Ptr->add_to_task_manager( task_mgr_Ptr );
        //proj_action_task_Ptr->add_to_task_manager( task_mgr_Ptr );
        //task_mgr_Ptr->add( move_action_task_Ptr );
        task_mgr_Ptr->add( proj_action_task_Ptr );
    }
 }
 
 #endif // PROJECTILE_ATTACK_INFO_H